<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Pj_WPDB_Session_Handler
 * https://github.com/pressjitsu/wpdb-phpsessions
 * Description: Use $wpdb to store $_SESSION data.
 *
 * @class Pj_WPDB_Session_Handler
 * @package Inventor/Classes
 * @author pressjitsu
 */

class Pj_WPDB_Session_Handler {
    public static $instance = null;

    public $wpdb = null;
    public $table = 'pj_wpdb_sessions';
    public $version = 1;

    /**
     * Open a session.
     */
    public function open() {
        return true;
    }

    /**
     * Close a session.
     */
    public function close() {
        return true;
    }

    /**
     * Read session data.
     *
     * @param string $id Session id.
     * @return mixed Session data or null.
     */
    public function read( $id ) {
        if ( ! $this->wpdb )
            return null;

        return @$this->wpdb->get_var( $this->wpdb->prepare( "SELECT `data` FROM `{$this->table}` WHERE `id` = %s;", $id ) );
    }

    /**
     * Write a session.
     *
     * @param string $id Session id.
     * @param string $data Session data (serialized for session storage).
     * @return mixed
     */
    public function write( $id, $data ) {
        if ( ! $this->wpdb )
            return null;

        return @$this->wpdb->query( $this->wpdb->prepare( "REPLACE INTO `{$this->table}` VALUES ( %s, %s, %d );", $id, $data, time() ) );
    }

    /**
     * Destroy a session.
     *
     * @param string $id Session id.
     * @return bool
     */
    public function destroy( $id ) {
        if ( ! $this->wpdb )
            return false;

        return (bool) @$this->wpdb->query( $this->wpdb->prepare( "DELETE FROM `{$this->table}` WHERE `id` = %s;", $id ) );
    }

    /**
     * Garbage collection.
     */
    public function gc( $max ) {
        return true;
    }

    /**
     * Compare versions and maybe run an upgrade routine.
     */
    public function maybe_upgrade() {
        $current_version = (int) get_site_option( 'pj_wpdb_sessions_version', 0 );
        if ( version_compare( $this->version, $current_version, '>' ) )
            $this->do_upgrade( $current_version );
    }

    /**
     * Perform an upgrade routine.
     *
     * @param int $current_version The version number from which to perform the upgrades.
     */
    public function do_upgrade( $current_version ) {
        global $wpdb;

        if ( $current_version < 1 ) {
            @$wpdb->query( "CREATE TABLE `{$this->table}` (
				`id` varchar(255) NOT NULL,
				`data` mediumtext NOT NULL,
				`timestamp` int(255) NOT NULL,
				PRIMARY KEY (`id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8;" );

            $current_version = 1;
            update_site_option( 'pj_wpdb_sessions_version', $current_version );
        }
    }

    /**
     * Cron-powered garbage collection.
     */
    public function cron_gc() {
        global $wpdb;
        @$wpdb->query( $wpdb->prepare( "DELETE FROM `{$this->table}` WHERE `timestamp` < %d;", time() - HOUR_IN_SECONDS * 24 ) );
    }

    /**
     * Runs at the end of this script.
     */
    public static function init() {
        if ( ! self::$instance ) {
            self::$instance = new Pj_WPDB_Session_Handler;
            self::$instance->wpdb = clone $GLOBALS['wpdb'];
            self::$instance->maybe_upgrade();

            session_set_save_handler(
                array( self::$instance, 'open' ),
                array( self::$instance, 'close' ),
                array( self::$instance, 'read' ),
                array( self::$instance, 'write' ),
                array( self::$instance, 'destroy' ),
                array( self::$instance, 'gc' )
            );

            register_shutdown_function( 'session_write_close' );

            if ( ! wp_next_scheduled( 'pj_wpdb_sessions_gc' ) )
                wp_schedule_event( time(), 'hourly', 'pj_wpdb_sessions_gc' );

            add_action( 'pj_wpdb_sessions_gc', array( self::$instance, 'cron_gc' ) );
        }

        return self::$instance;
    }

    // No outsiders.
    private function __construct() {}
}

Pj_WPDB_Session_Handler::init();