/*!
 * jQuery Google Map
 *
 * @author Pragmatic Mates, http://pragmaticmates.com
 * @version 1.1
 * @license GPL 2
 * @link https://github.com/PragmaticMates/jquery-google-map
 */

if ( ! gettext ) {
	var gettext = {
		loading_address: "Loading address...",
		failed_to_load_address: "Failed to load address",
		your_current_location: "Your current location"
	};
}

(function ($) {
	var settings;
	var element;
	var map;
	var markers = new Array();
	var markerCluster;
	var clustersOnMap = new Array();
	var clusterListener;

	var methods = {
		init: function (options) {
			element = $(this);

			var defaults = $.extend({
				fitBounds: false,
				geolocation: false,
				styles: null,
				zoom: 14,
				markers: [],
				infowindow: {
					borderBottomSpacing: 6,
					width: 340,
					height: 150
				},
				marker: {
					height: 40,
					width: 40
				},
				cluster: {
					height: 40,
					width: 40,
					gridSize: 40
				},
                zoomControl: false,
                mapTypeId: google.maps.MapTypeId.ROADMAP
			});

			settings = $.extend({}, defaults, options);

			loadMap();
			if (options.callback) {
				options.callback();
			}

			google.maps.Map.prototype.setCenterWithOffset = function (latlng, offsetX, offsetY) {
				var map = this;
				var ov = new google.maps.OverlayView();

				ov.onAdd = function () {
					var proj = this.getProjection();
					var aPoint = proj.fromLatLngToContainerPixel(latlng);
					aPoint.x = aPoint.x + offsetX;
					aPoint.y = aPoint.y + offsetY;
					map.setCenter(proj.fromContainerPixelToLatLng(aPoint));
				};

				ov.draw = function () {};
				ov.setMap(this);
			};

			return $(this);
		},

		removeMarkers: function () {
			if (markers) {
				for (i = 0; i < markers.length; i++) {
					if (markers[i].infobox !== undefined) {
						markers[i].infobox.close();
						markers[i].marker.close();
						markers[i].setMap(null);
					}
				}

				markers = [];

				markerCluster.clearMarkers();

				$.each(clustersOnMap, function (index, cluster) {
					cluster.cluster.close();
				});

				clusterListener.remove();
			}
		},

		addMarkers: function (options) {
			settings.markers = options.markers;
			renderElements(options.fitBounds);
		}
	};

	$.fn.google_map = function (method) {
		if (methods[method]) {
			return methods[ method ].apply(this, Array.prototype.slice.call(arguments, 1));
		} else if (typeof method === 'object' || !method) {
			return methods.init.apply(this, arguments);
		} else {
			$.error('Method ' + method + ' does not exist on Inventor Map');
		}
	};

	function loadMap() {
		var mapOptions = {
			zoom: settings.zoom,
			styles: settings.styles,
			mapTypeId: settings.mapTypeId,
			scrollwheel: false,
			draggable: true,
			mapTypeControl: true,
			panControl: false,
			zoomControl: settings.zoomControl
		};

		if (settings.center !== undefined) {
			mapOptions.center = new google.maps.LatLng(settings.center.latitude, settings.center.longitude);
		}

		map = new google.maps.Map($(element)[0], mapOptions);

		if (settings.geolocation) {
			if (navigator.geolocation) {
				$('input[name=geolocation]').attr('value', gettext.loading_address);

				navigator.geolocation.getCurrentPosition(function (position) {
					initialLocation = new google.maps.LatLng(position.coords.latitude, position.coords.longitude);
					map.setCenter(initialLocation);

					$('input[name=latitude]').attr('value', position.coords.latitude);
					$('input[name=longitude]').attr('value', position.coords.longitude);
					addressFromPosition(position);
				}, function () {
					map.center = new google.maps.LatLng(settings.center.latitude, settings.center.longitude);
				});
			} else {
				map.center = new google.maps.LatLng(settings.center.latitude, settings.center.longitude);
				$('input[name=geolocation]').attr('value', gettext.failed_to_load_address);
			}
		}

		if (settings.center === undefined || settings.fitBounds) {
			if (settings.markers.length !== 0) {
				var bound = new google.maps.LatLngBounds();
				for (var i in settings.markers) {
					bound.extend(new google.maps.LatLng(settings.markers[i].latitude, settings.markers[i].longitude));
				}
				map.fitBounds(bound);
			}
		}

		google.maps.event.addListener(map, 'zoom_changed', function () {
			$.each(markers, function (index, marker) {
				if (marker.infobox !== undefined) {
					marker.infobox.close();
					marker.infobox.isOpen = false;
				}
			});
		});

		var geolocation_field = $('.map-content .form-control-geolocation');

		if (geolocation_field.length !== 0) {
			var autocomplete = new google.maps.places.Autocomplete(geolocation_field[0]);

			geolocation_field.keypress(function (e) {
				if (13 === e.keyCode) {
					e.preventDefault();

					//google.maps.event.addListener(autocomplete, 'place_changed', function() {
					//	geolocation_field.closest('form').submit();
					//});
				}
			});

			google.maps.event.addListener(autocomplete, 'place_changed', function () {
				var place = autocomplete.getPlace();

				var map_dom = geolocation_field.closest(':has(#map)').find('#map');
				var fitBounds = map_dom.data('fit-bounds');

				if ( place && place.geometry ) {
					var latitude = $('input[name=latitude]');
					var longitude = $('input[name=longitude]');

					if (place.geometry.viewport) {
						map.fitBounds(place.geometry.viewport);
					} else {
						map.setCenter(place.geometry.location);
						//map.setZoom(17);
					}

					latitude.val(place.geometry.location.lat());
					longitude.val(place.geometry.location.lng());

					map_dom.data('fit-bounds', false);
				}

				geolocation_field.closest('form').submit();
				map_dom.data('fit-bounds', fitBounds);
			});
		}

		// Add polygon to map
		if (settings.polygon) {
			settings.polygon.setMap(map);

			if (settings.fitBounds) {
				var path = settings.polygon.getPath().getArray();

				for (var i in path) {
					var lat = path[i].lat();
					var lng = path[i].lng();
					bound.extend(new google.maps.LatLng(lat, lng));
				}
				map.fitBounds(bound);
			}
		}

        initControls();
		renderElements(false);
	}

    function initControls() {
        $('#map-control-zoom-in').on('click', function(e) {
            e.preventDefault();
            var zoom = map.getZoom();
            map.setZoom(zoom + 1);
        });

        $('#map-control-zoom-out').on('click', function(e) {
            e.preventDefault();
            var zoom = map.getZoom();
            map.setZoom(zoom - 1);
        });

        $('#map-control-type-roadmap').on('click', function(e) {
            e.preventDefault();
            map.setMapTypeId(google.maps.MapTypeId.ROADMAP);
        });

        $('#map-control-type-terrain').on('click', function(e) {
            e.preventDefault();
            map.setMapTypeId(google.maps.MapTypeId.TERRAIN);
        });

        $('#map-control-type-satellite').on('click', function(e) {
            e.preventDefault();
            map.setMapTypeId(google.maps.MapTypeId.SATELLITE);
        });

        $('#map-control-current-position').on('click', function(e) {
            e.preventDefault();

			$('input[name=geolocation]').attr('value', gettext.loading_address);

			navigator.geolocation.getCurrentPosition(function (position) {
                initialLocation = new google.maps.LatLng(position.coords.latitude, position.coords.longitude);
				map.setCenter(initialLocation);

				$('input[name=latitude]').attr('value', position.coords.latitude);
				$('input[name=longitude]').attr('value', position.coords.longitude);

				// update geolocation input with current address
				addressFromPosition(position);

				// add marker of current position into the map
				new google.maps.Marker({
					position: initialLocation,
					map: map,
					title: gettext.your_current_location
				});

			}, function (error) {
                map.center = new google.maps.LatLng(settings.center.latitude, settings.center.longitude);
				$('input[name=geolocation]').attr('value', gettext.failed_to_load_address);
            });
        });
    }

	function addressFromPosition(position) {
		var geolocation_field = $('input[name=geolocation]');

		var geocoder = new google.maps.Geocoder();
		var latlng = {lat: position.coords.latitude, lng: position.coords.longitude};
		geocoder.geocode({'location': latlng}, function(results, status) {
			var address = '';
			if (status === 'OK') {
				address = results[0].formatted_address;
				var map_dom = geolocation_field.closest(':has(#map)').find('#map');

				var fitBounds = map_dom.data('fit-bounds');
				map_dom.data('fit-bounds', false);
				geolocation_field.closest('form').submit();
				map_dom.data('fit-bounds', fitBounds);
			}
			geolocation_field.attr('value', address);
		});
	}

	function isClusterOnMap(clustersOnMap, cluster) {
		if (cluster === undefined) {
			return false;
		}

		if (clustersOnMap.length == 0) {
			return false;
		}

		var val = false;

		$.each(clustersOnMap, function (index, cluster_on_map) {
			if (cluster_on_map.getCenter() == cluster.getCenter()) {
				val = cluster_on_map;
			}
		});

		return val;
	}

	function addClusterOnMap(cluster) {
		var newCluster = new InfoBox({
			markers: cluster.getMarkers(),
			draggable: true,
			content: '<div class="clusterer"><div class="clusterer-inner">' + cluster.getMarkers().length + '</div></div>',
			disableAutoPan: true,
			pixelOffset: new google.maps.Size(-21, -21),
			position: cluster.getCenter(),
			closeBoxURL: "",
			isHidden: false,
			enableEventPropagation: true,
			pane: "mapPane"
		});

		cluster.cluster = newCluster;

		cluster.markers = cluster.getMarkers();
		cluster.cluster.open(map, cluster.marker);
		clustersOnMap.push(cluster);
	}

	function renderElements(fitBounds) {
		var bounds = new google.maps.LatLngBounds();

        $.each(clustersOnMap, function (index, cluster) {
            cluster.cluster.close();
        });

		$.each(settings.markers, function (index, markerObject) {
			bounds.extend(new google.maps.LatLng(markerObject.latitude, markerObject.longitude));

			// Create invisible markers on the map
			var args = {
				position: new google.maps.LatLng(markerObject.latitude, markerObject.longitude),
				map: map
			};

			if (markerObject.marker_content) {
				args['content'] = markerObject.marker_content;
				args['shadow'] = 0;
				args['anchor'] = RichMarkerPosition.MIDDLE;
			}

			// Create marker
			var marker = new RichMarker(args);

			// Create infobox for infowindow
			if (markerObject.content) {
				marker.infobox = new InfoBox({
					content: markerObject.content,
					disableAutoPan: true,
					pixelOffset: new google.maps.Size(-settings.infowindow.width/2, -settings.infowindow.height-settings.marker.height/2),
					position: new google.maps.LatLng(markerObject.latitude, markerObject.longitude),
					isHidden: false,
                    closeBoxURL: "",
					pane: "floatPane",
					enableEventPropagation: false
				});

				marker.infobox.isOpen = false;
			}

			// Handle logic for opening/closing info windows
			markers.push(marker);

			google.maps.event.addListener(marker, 'click', function (e) {
				if (marker.infobox !== undefined) {
                    var curMarker = this;
                    var position = new google.maps.LatLng(marker.getPosition().lat(), marker.getPosition().lng());

                    map.panTo(position);
                    map.panBy(0, -(settings.infowindow.height/2 + 20));

                    $.each(markers, function (index, marker) {
                        if (marker !== curMarker) {
                            marker.infobox.close();
                            marker.infobox.isOpen = false;
                        }
                    });

                    if (curMarker.infobox) {
                        if (curMarker.infobox.isOpen === false) {
                            curMarker.infobox.open(map, this);
                            curMarker.infobox.isOpen = true;
                        } else {
                            curMarker.infobox.close();
                            curMarker.infobox.isOpen = false;
                        }
                    }
                }
			});
		});

		markerCluster = new MarkerClusterer(map, markers, {
            gridSize: settings.cluster.gridSize,
			styles: [
				{
					textColor: 'transparent',
					height: settings.cluster.height,
					url: settings.transparentClusterImage,
					width: settings.cluster.width
				}
			]
		});

		clustersOnMap = new Array();
		clusterListener = google.maps.event.addListener(markerCluster, 'clusteringend', function (clusterer) {
			var availableClusters = clusterer.getClusters();
			var activeClusters = new Array();

			$.each(availableClusters, function (index, cluster) {
				if (cluster.getMarkers().length > 1) {
					activeClusters.push(cluster);
				}
			});

			$.each(availableClusters, function (index, cluster) {
				if (cluster.getMarkers().length > 1) {
					var val = isClusterOnMap(clustersOnMap, cluster);

					if (val !== false) {
						val.cluster.setContent('<div class="clusterer"><div class="clusterer-inner">' + cluster.getMarkers().length + '</div></div>');
						val.markers = cluster.getMarkers();
					} else {
						addClusterOnMap(cluster);
					}
				} else {
					// Remove old cluster
					$.each(clustersOnMap, function (index, cluster_on_map) {
						if (cluster !== undefined && cluster_on_map !== undefined) {
							if (cluster_on_map.getCenter() == cluster.getCenter()) {
								// Show all cluster's markers
								cluster_on_map.cluster.close();
								clustersOnMap.splice(index, 1);
							}
						}
					});
				}
			});

			var newClustersOnMap = new Array();

			$.each(clustersOnMap, function (index, clusterOnMap) {
				var remove = true;

				$.each(availableClusters, function (index2, availableCluster) {
					if (availableCluster.getCenter() == clusterOnMap.getCenter()) {
						remove = false;
					}
				});

				if (!remove) {
					newClustersOnMap.push(clusterOnMap);
				} else {
					clusterOnMap.cluster.close();
				}
			});

			clustersOnMap = newClustersOnMap;
		});

		if (fitBounds === true && markers.length > 0) {
			map.fitBounds(bounds);
		}

        $(document).on('click touchstart', '.infobox .close', function(e) {
            e.preventDefault();

            $.each(markers, function(index, marker) {
                marker.infobox.isHidden = true;
                marker.infobox.close();
            });
        });
	}
})(jQuery);
